<?php
namespace Elementor;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly
 
/**
 * RyanCV Clients Widget.
 *
 * @since 1.0
 */
class RyanCV_Clients_Widget extends Widget_Base {

	public function get_name() {
		return 'ryancv-clients';
	}

	public function get_title() {
		return esc_html__( 'Clients', 'ryancv-plugin' );
	}

	public function get_icon() {
		return 'fab fa-cuttlefish';
	}

	public function get_categories() {
		return [ 'ryancv-category' ];
	}

	/**
	 * Register widget controls.
	 *
	 * @since 1.0
	 */
	protected function _register_controls() {

		$this->start_controls_section(
			'title_tab',
			[
				'label' => esc_html__( 'Title', 'ryancv-plugin' ),
				'tab' => Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'title',
			[
				'label'       => esc_html__( 'Title', 'ryancv-plugin' ),
				'type'        => Controls_Manager::TEXTAREA,
				'placeholder' => esc_html__( 'Enter title', 'ryancv-plugin' ),
				'default'     => esc_html__( 'Title', 'ryancv-plugin' ),
			]
		);

		$this->add_control(
			'title_tag',
			[
				'label'       => esc_html__( 'Title Tag', 'ryancv-plugin' ),
				'type'        => Controls_Manager::SELECT,
				'default' => 'h2',
				'options' => [
					'h1'  => __( 'H1', 'ryancv-plugin' ),
					'h2' => __( 'H2', 'ryancv-plugin' ),
					'div' => __( 'DIV', 'ryancv-plugin' ),
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'content_tab',
			[
				'label' => esc_html__( 'Content', 'ryancv-plugin' ),
				'tab' => Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'items',
			[
				'label' => esc_html__( 'Items', 'ryancv-plugin' ),
				'type' => Controls_Manager::REPEATER,
				'prevent_empty' => false,
				'fields' => [
					[
						'name' => 'image',
						'label' => esc_html__( 'Image', 'ryancv-plugin' ),
						'type' => Controls_Manager::MEDIA,
						'default' => [
							'url' => \Elementor\Utils::get_placeholder_image_src(),
						],
					],
					[
						'name' => 'name',
						'label'       => esc_html__( 'Name', 'ryancv-plugin' ),
						'type'        => Controls_Manager::TEXTAREA,
						'placeholder' => esc_html__( 'Enter name', 'ryancv-plugin' ),
						'default'	=> esc_html__( 'Enter name', 'ryancv-plugin' ),
					],
					[
						'name' => 'link',
						'label' => esc_html__( 'URL', 'ryancv-plugin' ),
						'type' => Controls_Manager::URL,
						'show_external' => true,
					],
				],
				'title_field' => '{{{ name }}}',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'title_styling',
			[
				'label'     => esc_html__( 'Title', 'ryancv-plugin' ),
				'tab'       => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'title_color',
			[
				'label'     => esc_html__( 'Color', 'ryancv-plugin' ),
				'type'      => Controls_Manager::COLOR,
				'default'	=> '',
				'selectors' => [
					'{{WRAPPER}} .section .titles .title' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'title_typography',
				'selector' => '{{WRAPPER}} .section .titles .title',
			]
		);

		$this->end_controls_section();
	}


	/**
	 * Render widget output on the frontend.
	 *
	 * @since 1.0
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();
		$this->add_inline_editing_attributes( 'title', 'basic' );
		?>

		<!-- Clients -->
		<div class="content clients">

			<?php if ( $settings['title'] ) : ?>
			<!-- title -->
			<<?php echo esc_attr( $settings['title_tag'] ); ?> class="title">
				<span <?php echo $this->get_render_attribute_string( 'title' ); ?>><?php echo wp_kses_post( $settings['title'] ); ?></span>
			</<?php echo esc_attr( $settings['title_tag'] ); ?>>
			<?php endif; ?>

			<?php if ( $settings['items'] ) : ?>
			<!-- content -->
			<div class="row client-items">

				<?php foreach ( $settings['items'] as $index => $item ) : ?>
				<?php if ( $item['image'] ) : $image = wp_get_attachment_image_url( $item['image']['id'], 'ryancv_92x92' ); ?>
				<!-- client item -->
				<div class="col col-d-3 col-t-3 col-m-6 border-line-v">
					<div class="client-item">
						<div class="image">
							<?php if ( $item['link'] ) : ?>
							<a <?php if ( $item['link']['is_external'] ) : ?> target="_blank"<?php endif; ?><?php if ( $item['link']['nofollow'] ) : ?> rel="nofollow"<?php endif; ?> href="<?php echo esc_url( $item['link']['url'] ); ?>">
							<?php endif; ?>
								<img src="<?php echo esc_url( $image ); ?>" alt="<?php echo esc_attr( $item['name'] ); ?>" />
							<?php if ( $item['link'] ) : ?>
							</a>
							<?php endif; ?>
						</div>
					</div>
				</div>
				<?php endif; ?>
				<?php endforeach; ?>

				<div class="clear"></div>
			</div>
			<?php endif; ?>
		</div>

		<?php
	}

	/**
	 * Render widget output in the editor.
	 *
	 * Written as a Backbone JavaScript template and used to generate the live preview.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function _content_template() {
		?>
		<#
		view.addInlineEditingAttributes( 'title', 'basic' );
		#>
		
		<!-- Clients -->
		<div class="content clients">

			<# if ( settings.title ) { #>
			<!-- title -->
			<{{{ settings.title_tag }}} class="title">
				<span {{{ view.getRenderAttributeString( 'title' ) }}}>{{{ settings.title }}}</span>
			</{{{ settings.title_tag }}}>
			<# } #>

			<# if ( settings.items ) { #>
			<!-- content -->
			<div class="row client-items">
				<# _.each( settings.items, function( item, index ) { #>
				<# if ( item.image ) { #>
				<!-- client item -->
				<div class="col col-d-3 col-t-3 col-m-6 border-line-v">
					<div class="client-item">
						<div class="image">
							<# if ( item.link ) { #>
							<a <# if ( item.link.is_external ) { #> target="_blank"<# } #><# if ( item.link.nofollow ) { #> rel="nofollow"<# } #> href="{{{ item.link.url }}}">
							<# } #>
								<img src="{{{ item.image.url }}}" alt="{{{ item.name }}}" />
							<# if ( item.link ) { #>
							</a>
							<# } #>
						</div>
					</div>
				</div>
				<# } #>
				<# }); #>

				<div class="clear"></div>
			</div>
			<# } #>
		</div>

		<?php 
	}
}

Plugin::instance()->widgets_manager->register_widget_type( new RyanCV_Clients_Widget() );